<?php

const AMWALPAY_VERSION = '1.0.0';

class ControllerExtensionPaymentAmwalPay extends Controller
{
    private $id = 'amwalpay';
    private $error = [];
    private $fields = [
        'status',
        'live',
        'merchant_id',
        'terminal_id',
        'secret_key',
        'complete_paid_order',
        'sort_order',
        'debug_order',
    ];

    /**
     * Loads the payment module configuration page in the OpenCart admin panel.
     * Handles settings updates and displays success/error messages.
     */
    public function index()
    {
        $path = "extension/payment/$this->id";

        // Load model
        $this->load->model($path);

        // Load language file
        $data = $this->language->load($path);

        // Determine OpenCart version compatibility
        if (version_compare(VERSION, '3.0.0', '<')) {
            $userToken = 'token=' . $this->session->data['token'];
            $extensionLink = 'extension/extension&type=payment';
            $paymentCode = $data['ocCode'] = $this->id;
        } else {
            $userToken = 'user_token=' . $this->session->data['user_token'];
            $extensionLink = 'marketplace/extension&type=payment';
            $paymentCode = $data['ocCode'] = 'payment_' . $this->id;
        }

        // Set document title
        $this->document->setTitle($this->language->get('heading_title'));

        // Remove error and success messages on reload
        if (isset($this->session->data['error_warning'])) {
            $data['error_warning'] = $this->session->data['error_warning'];
            unset($this->session->data['error_warning']);
        }

        if (isset($this->session->data['success'])) {
            $data['success'] = $this->session->data['success'];
            unset($this->session->data['success']);
        }

        // Handle form submission
        if ($this->request->server['REQUEST_METHOD'] == 'POST') {
            try {
                $this->load->model('setting/setting');

                // Sanitize input values
                $postData = $this->request->post;
                foreach ($postData as $key => $value) {
                    $postData[$key] = $this->db->escape(trim($value));
                }

                // Convert checkbox values to 0 or 1
                $postData[$paymentCode . '_debug_order'] = isset($postData[$paymentCode . '_debug_order']) ? 1 : 0;
                $postData[$paymentCode . '_complete_paid_order'] = isset($postData[$paymentCode . '_complete_paid_order']) ? 1 : 0;

                // Validate required fields
                if (!empty($postData[$paymentCode . '_merchant_id']) &&
                    !empty($postData[$paymentCode . '_terminal_id']) &&
                    !empty($postData[$paymentCode . '_secret_key']) &&
                    !empty($postData[$paymentCode . '_live'])
                ) {
                    // Save settings
                    $this->model_setting_setting->editSetting($paymentCode, $postData);

                    // Set success message
                    $this->session->data['success'] = $this->language->get('text_success');

                    // Redirect to the same page
                    $this->response->redirect($this->url->link($path, $userToken, true));
                } else {
                    $this->error['warning'] = $this->language->get('invalid_amwal_information');
                }
            } catch (Exception $exc) {
                $this->error['warning'] = $exc->getMessage();
            }
        }

        // Pass variables to the view
        $data['paymentCode'] = $paymentCode;
        $data['error_danger'] = $this->error['warning'] ?? '';
        $data['action'] = $this->url->link($path, $userToken, true);
        $data['cancel'] = $this->url->link($extensionLink, $userToken, true);

        // Breadcrumbs
        $data['breadcrumbs'] = [
            [
                'text' => $this->language->get('text_home'),
                'href' => $this->url->link('common/dashboard', $userToken, true)
            ],
            [
                'text' => $this->language->get('text_extension'),
                'href' => $data['cancel']
            ],
            [
                'text' => $this->language->get('heading_title'),
                'href' => $data['action']
            ]
        ];

        // Load stored settings
        foreach ($this->fields as $field) {
            $key = $paymentCode . '_' . $field;
            $data[$field] = $this->config->get($key);
        }

        // Load additional settings
        $data['live'] = $this->config->get($paymentCode . '_live') ?? 'uat';
        $data['debug_order'] = $this->config->get($paymentCode . '_debug_order') ?? 1;
        $data['complete_paid_order'] = $this->config->get($paymentCode . '_complete_paid_order');
        $data['log_dir'] = DIR_LOGS;

        // Default values
        if (empty($data['sort_order'])) {
            $data['sort_order'] = 1;
        }

        if (empty($data['debug'])) {
            $data['debug'] = 1;
        }

        // Load styles
        $this->document->addStyle('view/stylesheet/amwalpay/admin.css');
        $data['styles'] = $this->document->getStyles();
        $data['adminURL'] = HTTPS_SERVER;

        // Load common UI components
        $data['header'] = $this->load->controller('common/header');
        $data['column_left'] = $this->load->controller('common/column_left');
        $data['footer'] = $this->load->controller('common/footer');

        // Render the settings page
        $this->response->setOutput($this->load->view($path, $data));
    }

    /**
     * Installs the payment module and sets default configuration values.
     */
    public function install()
    {
        $paymentCode = version_compare(VERSION, '3.0.0', '<') ? '' : 'payment_';

        // Set default values for fields
        $data = [];
        foreach ($this->fields as $field) {
            $data[$paymentCode . $this->id . '_' . $field] = $this->config->get($paymentCode . '_' . $field);
        }

        // Save settings
        $this->load->model('setting/setting');
        $this->model_setting_setting->editSetting($paymentCode . $this->id, $data);
    }
}
