<?php

define('AMWALPAY_OC3_23_VERSION', '1.0.0');


class ControllerExtensionPaymentAmwalPay extends Controller
{

    protected $id;
    protected $amwalDir;
    protected $paymentCode;

    protected $orderInfo;
    protected $merchant_id;
    protected $terminal_id;
    protected $secret_key;
    protected $live;
    protected $complete_paid_order;

    /**
     * Constructor initializes the payment settings and configurations.
     */
    public function __construct($registry)
    {
        parent::__construct($registry);
        $this->id = 'amwalpay';
        $this->load->model('checkout/order');
        $this->load->model('extension/payment/amwalpay');
        $this->amwalDir = 'extension/payment/' . $this->id;
        $this->paymentCode = (version_compare(VERSION, '3.0.0', '<')) ? $this->id : 'payment_' . $this->id;
        $this->merchant_id = $this->config->get($this->paymentCode . '_merchant_id');
        $this->terminal_id = $this->config->get($this->paymentCode . '_terminal_id');
        $this->secret_key = $this->config->get($this->paymentCode . '_secret_key');
        $this->live = $this->config->get($this->paymentCode . '_live');
        $this->complete_paid_order = $this->config->get($this->paymentCode . '_complete_paid_order');

    }

    /**
     * Loads the payment view.
     */
    public function index()
    {
        $data = $this->language->load($this->amwalDir);
        $data['amwalpayVersion'] = AMWALPAY_OC3_23_VERSION;
        $data['action'] = 'index.php?route=' . $this->amwalDir . '/execute_smartbox';

        return $this->load->view($this->amwalDir, $data);
    }

    /**
     * Executes the payment process by generating a payment request.
     */
    public function execute_smartbox()
    {
        if (!isset($this->session->data['order_id'])) {
            return $this->redirectToErrorPage('Session has been expired please try again later');
        }
        $orderId = $this->session->data['order_id'];
        //check for order object
        $this->load->model('checkout/order');
        if (empty($this->model_checkout_order->getOrder($orderId))) {
            return $this->redirectToErrorPage('Order is not exist, please contact site admin! Or, you can logout and try again later.');
        }

        $this->load->model('account/order');
        $this->orderInfo = $this->model_checkout_order->getOrder($orderId);
        $this->model_checkout_order->addOrderHistory($orderId, 1, "Pending AmwalPay payment", true);
        $amount = $this->orderInfo['total'];

        $locale = $this->session->data['language']; // Get the current language locale

        $datetime = date('YmdHis');

        // if $locale content en make $locale = "en"
        if (strpos($locale, 'en') !== false) {
            $locale = "en";
        } else {
            $locale = "ar";
        }
        $refNumber = $orderId . '_' . date("ymds");
        // Generate secure hash
        $secret_key = self::generateString(
            $amount,
            512,
            $this->merchant_id,
            $refNumber
            ,
            $this->terminal_id,
            $this->secret_key,
            $datetime
        );

        $data = [
            'AmountTrxn' => "$amount",
            'MerchantReference' => "$refNumber",
            'MID' => $this->merchant_id,
            'TID' => $this->terminal_id,
            'CurrencyId' => 512,
            'LanguageId' => $locale,
            'SecureHash' => $secret_key,
            'TrxDateTime' => $datetime,
            'PaymentViewType' => 1,
            'RequestSource' => 'Checkout_Open_Cart',
            'SessionToken' => '',
        ];
        $smartBoxUrl = $this->getSmartBoxUrl();
        $jsonData = json_encode($data);
        $json['jsonData'] = $jsonData;
        $json['url'] = $smartBoxUrl;
        $json['callback_url'] = $this->config->get('config_url') . "index.php?route=extension/payment/amwalpay/callback";
        $json['base_url'] = $this->config->get('config_url');
        self::addLogs($this->config->get($this->paymentCode . '_debug_order'), $this->config->get($this->paymentCode . '_log_path'), 'Payment Request: ', print_r($json, 1));
        $this->response->setOutput($this->load->view('common/smartbox', $json));


    }
    /**
     * Returns the appropriate SmartBox URL based on the environment.
     */
    public function getSmartBoxUrl()
    {
        if ($this->live == "prod") {
            $liveurl = "https://checkout.amwalpg.com/js/SmartBox.js?v=1.1";

        } else if ($this->live == "uat") {
            $liveurl =
                "https://test.amwalpg.com:7443/js/SmartBox.js?v=1.1";

        } else if ($this->live == "sit") {
            $liveurl =
                "https://test.amwalpg.com:19443/js/SmartBox.js?v=1.1";
        }
        return $liveurl;
    }
    /**
     * Generates a secure hash for transaction validation.
     */
    public static function generateString(
        $amount,
        $currencyId,
        $merchantId,
        $merchantReference,
        $terminalId,
        $hmacKey,
        $trxDateTime
    ) {

        $string = "Amount={$amount}&CurrencyId={$currencyId}&MerchantId={$merchantId}&MerchantReference={$merchantReference}&RequestDateTime={$trxDateTime}&SessionToken=&TerminalId={$terminalId}";

        $sign = self::encryptWithSHA256($string, $hmacKey);
        return strtoupper($sign);
    }
    /**
     * Encrypts a given string using SHA-256 HMAC.
     */
    public static function encryptWithSHA256($input, $hexKey)
    {
        // Convert the hex key to binary
        $binaryKey = hex2bin($hexKey);
        // Calculate the SHA-256 hash using hash_hmac
        $hash = hash_hmac('sha256', $input, $binaryKey);
        return $hash;
    }
    /**
     * Generates a signed string for filtering data.
     *
     */
    public static function generateStringForFilter(
        $data,
        $hmacKey

    ) {
        // Convert data array to string key value with and sign
        $string = '';
        foreach ($data as $key => $value) {
            $string .= $key . '=' . ($value === "null" || $value === "undefined" ? '' : $value) . '&';
        }
        $string = rtrim($string, '&');
        // Generate SIGN
        $sign = self::encryptWithSHA256($string, $hmacKey);
        return strtoupper($sign);
    }

    /**
     * Redirects the user to an error page with a message.
     */

    function redirectToErrorPage($errMessage)
    {
        $this->session->data['error'] = $errMessage;

        if ($this->config->get('quickcheckout_status') == 1) {
            $this->response->redirect($this->url->link('checkout/cart', '', true));
        }
        $this->response->redirect($this->url->link('checkout/checkout', '', true));
    }

    /**
     * Execute the callback process after payment
     *
     */
    public function callback()
    {
        $merchantReference = isset($this->request->get['merchantReference']) ? htmlspecialchars($this->request->get['merchantReference']) : '';
        $amount = isset($this->request->get['amount']) ? htmlspecialchars($this->request->get['amount']) : '';
        $currencyId = isset($this->request->get['currencyId']) ? htmlspecialchars($this->request->get['currencyId']) : '';
        $customerId = isset($this->request->get['customerId']) ? htmlspecialchars($this->request->get['customerId']) : '';
        $customerTokenId = isset($this->request->get['customerTokenId']) ? htmlspecialchars($this->request->get['customerTokenId']) : '';
        $responseCode = isset($this->request->get['responseCode']) ? htmlspecialchars($this->request->get['responseCode']) : '';
        $transactionId = isset($this->request->get['transactionId']) ? htmlspecialchars($this->request->get['transactionId']) : '';
        $transactionTime = isset($this->request->get['transactionTime']) ? htmlspecialchars($this->request->get['transactionTime']) : '';
        $secureHashValueOld = isset($this->request->get['secureHashValue']) ? htmlspecialchars($this->request->get['secureHashValue']) : '';

        $orderId = substr($merchantReference, 0, -9);
        $order = $this->validateOrderInfo($orderId);

        $isPaymentApproved = false;
        $integrityParameters = [
            "amount" => $amount,
            "currencyId" => $currencyId,
            "customerId" => $customerId,
            "customerTokenId" => $customerTokenId,
            "merchantId" => $this->merchant_id,
            "merchantReference" => $merchantReference,
            "responseCode" => $responseCode,
            "terminalId" => $this->terminal_id,
            "transactionId" => $transactionId,
            "transactionTime" => $transactionTime
        ];

        self::addLogs($this->config->get($this->paymentCode . '_debug_order'), $this->config->get($this->paymentCode . '_log_path'), 'Callback Response: ', print_r($integrityParameters, 1));
        $secureHashValue = self::generateStringForFilter($integrityParameters, $this->secret_key);
        $integrityParameters['secureHashValue'] = $secureHashValue;
        $integrityParameters['secureHashValueOld'] = $secureHashValueOld;

        if (($responseCode === '00' || $secureHashValue == $secureHashValueOld)) {
            $isPaymentApproved = true;
        }

        $info = 'Old Hash -- ' . $secureHashValueOld . '  New Hash -- ' . $secureHashValue;
        self::addLogs($this->config->get($this->paymentCode . '_debug_order'), $this->config->get($this->paymentCode . '_log_path'), $info);

        $info = '</br>Transaction ID: ' . '<b style="color:DodgerBlue;">' . $transactionId;

        if ($isPaymentApproved) {
            $note = 'AmwalPay : Payment Approved';
            $msg = 'In callback action, for order #' . $orderId . ' ' . $note;
            if ($this->complete_paid_order) {
                $status = 5;
            } else {
                $status = 2;
            }
            $this->model_checkout_order->addOrderHistory($orderId, $status, $note . $info, true);
            self::addLogs($this->config->get($this->paymentCode . '_debug_order'), $this->config->get($this->paymentCode . '_log_path'), $msg);
            $this->response->redirect($this->url->link('checkout/success', 'language=' . $this->config->get('config_language'), true));

        } else {
            $note = 'AmwalPay : Payment is not completed';
            $msg = 'In callback action, for order #' . $orderId . ' ' . $note;
            $this->model_checkout_order->addOrderHistory($orderId, 10, $note . $info, true);
            self::addLogs($this->config->get($this->paymentCode . '_debug_order'), $this->config->get($this->paymentCode . '_log_path'), $msg);
            $this->response->redirect($this->url->link('checkout/failure', 'language=' . $this->config->get('config_language'), true));
        }

    }

    /**
     * Validate order info.
     *
     */
    public function validateOrderInfo($orderId)
    {
        if (empty($orderId) || is_null($orderId) || $orderId === false || $orderId === "") {
            $this->redirectToErrorPage('Ops. you are accessing wrong order.');
        }
        $this->load->model('checkout/order');
        $order = $this->model_checkout_order->getOrder($orderId);

        if ($order['payment_code'] !== 'amwalpay') {
            $this->redirectToErrorPage('Ops. you are accessing wrong order.');
        }
        return $order;
    }
    /**
     * Adds logs to a specified file.
     *
     */
    public static function addLogs($debug, $file, $note, $data = false)
    {
        if (is_bool($data)) {
            ('1' === $debug) ? error_log(PHP_EOL . gmdate('d.m.Y h:i:s') . ' - ' . $note, 3, $file) : false;
        } else {
            ('1' === $debug) ? error_log(PHP_EOL . gmdate('d.m.Y h:i:s') . ' - ' . $note . ' -- ' . json_encode($data), 3, $file) : false;
        }
    }

}